<?php
final class Zibal {
  private const REQUEST_URL = 'https://gateway.zibal.ir/v1/request';
  private const VERIFY_URL  = 'https://gateway.zibal.ir/v1/verify';

  private string $merchant;
  private string $callback;

  public function __construct(string $merchant, string $callbackUrl) {
    $this->merchant = $merchant;
    $this->callback = $callbackUrl;
  }

  public function requestPayment(int $amount, string $orderId, ?string $mobile=null, string $description='سفارش فروشگاه'): array {
    $payload = [
      'merchant'    => $this->merchant,
      'amount'      => $amount,
      'callbackUrl' => $this->callback,
      'orderId'     => $orderId,
      'description' => $description,
    ];
    if ($mobile) $payload['mobile'] = $mobile;

    $resp = $this->postJson(self::REQUEST_URL, $payload);
    if (!isset($resp['trackId'])) {
      throw new RuntimeException('Zibal: invalid response '.json_encode($resp, JSON_UNESCAPED_UNICODE));
    }
    return $resp;
  }

  public function verify(string $trackId): array {
    $payload = ['merchant'=>$this->merchant, 'trackId'=>$trackId];
    return $this->postJson(self::VERIFY_URL, $payload);
  }

  public static function paymentStartUrl(string $trackId): string {
    return "https://gateway.zibal.ir/start/{$trackId}";
  }

  private function postJson(string $url, array $data): array {
    $ch = curl_init($url);
    curl_setopt_array($ch,[
      CURLOPT_POST=>true,
      CURLOPT_HTTPHEADER=>['Content-Type: application/json'],
      CURLOPT_POSTFIELDS=>json_encode($data, JSON_UNESCAPED_UNICODE),
      CURLOPT_RETURNTRANSFER=>true,
      CURLOPT_CONNECTTIMEOUT=>10,
      CURLOPT_TIMEOUT=>20,
    ]);
    $raw = curl_exec($ch);
    if ($raw === false) { throw new RuntimeException('Zibal CURL error: '.curl_error($ch)); }
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    $json = json_decode($raw,true) ?? [];
    if ($status>=400) { throw new RuntimeException('Zibal HTTP '.$status.': '.$raw); }
    return $json;
  }
}
