document.addEventListener("DOMContentLoaded", function () {
  function load(key) {
    try {
      const raw = localStorage.getItem(key);
      return raw ? JSON.parse(raw) : [];
    } catch (e) {
      return [];
    }
  }

  function save(key, value) {
    localStorage.setItem(key, JSON.stringify(value));
  }

  function getCart() {
    return load("cart");
  }

  function setCart(cart) {
    save("cart", cart);
    updateCartCount();
  }

  function addToCart(product, qty) {
    const cart = getCart();
    const delta = qty && qty > 0 ? qty : 1;
    const existing = cart.find((item) => item.id === product.id);
    if (existing) {
      existing.qty += delta;
    } else {
      cart.push({ ...product, qty: delta });
    }
    setCart(cart);
  }

  function removeFromCart(id) {
    const cart = getCart().filter((item) => item.id !== id);
    setCart(cart);
    renderCartPage();
  }

  function updateCartCount() {
    const cart = getCart();
    const count = cart.reduce((sum, item) => sum + item.qty, 0);
    const total = cart.reduce((sum, item) => sum + item.qty * (item.price || 0), 0);

    const el = document.getElementById("cart-count");
    if (el) el.textContent = count;

    const miniCount = document.getElementById("cart-count-mini");
    if (miniCount) miniCount.textContent = count;

    const miniTotal = document.getElementById("cart-total-mini");
    if (miniTotal) miniTotal.textContent = total.toLocaleString("fa-IR") + " تومان";
  }

  function getCompare() {
    return load("compare");
  }

  function setCompare(list) {
    save("compare", list);
    renderComparePage();
  }

  function addToCompare(product) {
    const list = getCompare();
    if (!list.find((p) => p.id === product.id)) {
      list.push(product);
      setCompare(list);
    }
  }

  function removeFromCompare(id) {
    const list = getCompare().filter((p) => p.id !== id);
    setCompare(list);
  }

  document.body.addEventListener("click", function (e) {
    const addCartBtn = e.target.closest(".add-to-cart");
    const addCompareBtn = e.target.closest(".add-to-compare");
    const removeCartBtn = e.target.closest(".remove-from-cart");
    const removeCompareBtn = e.target.closest(".remove-from-compare");
    const qtyPlus = e.target.closest(".qty-plus");
    const qtyMinus = e.target.closest(".qty-minus");

    if (addCartBtn) {
      const container =
        addCartBtn.closest("[data-product-id]") || addCartBtn.closest(".product-card");
      if (!container) return;
      let qty = 1;
      const qtyInput =
        container.querySelector(".qty-input") ||
        container.querySelector('input[name="qty"]');
      if (qtyInput) {
        const parsed = parseInt(qtyInput.value, 10);
        qty = !isNaN(parsed) && parsed > 0 ? parsed : 1;
      }
      const product = {
        id: container.dataset.productId,
        name: container.dataset.name,
        price: Number(container.dataset.price || 0),
      };
      addToCart(product, qty);
    }

    if (addCompareBtn) {
      const container =
        addCompareBtn.closest("[data-product-id]") || addCompareBtn.closest(".product-card");
      if (!container) return;
      const product = {
        id: container.dataset.productId,
        name: container.dataset.name,
        price: Number(container.dataset.price || 0),
      };
      addToCompare(product);
    }

    if (removeCartBtn) {
      const id = removeCartBtn.dataset.id;
      removeFromCart(id);
    }

    if (removeCompareBtn) {
      const id = removeCompareBtn.dataset.id;
      removeFromCompare(id);
      renderComparePage();
    }

    if (qtyPlus) {
      const wrap = qtyPlus.closest(".qty-control");
      const input = wrap && wrap.querySelector(".qty-input");
      if (input) {
        const current = parseInt(input.value, 10) || 1;
        input.value = current + 1;
      }
    }

    if (qtyMinus) {
      const wrap = qtyMinus.closest(".qty-control");
      const input = wrap && wrap.querySelector(".qty-input");
      if (input) {
        let current = parseInt(input.value, 10) || 1;
        current = current - 1;
        if (current < 1) current = 1;
        input.value = current;
      }
    }
  });

  function renderCartPage() {
    const body = document.getElementById("cart-body");
    const emptyEl = document.getElementById("cart-empty");
    const tableWrapper = document.getElementById("cart-table-wrapper");
    const summaryTotalEl = document.getElementById("cart-total-summary");

    if (!body || !emptyEl || !tableWrapper) return;

    const cart = getCart();

    if (!cart.length) {
      emptyEl.classList.remove("hidden");
      tableWrapper.classList.add("hidden");
      if (summaryTotalEl) summaryTotalEl.textContent = "۰ تومان";
      return;
    }

    emptyEl.classList.add("hidden");
    tableWrapper.classList.remove("hidden");

    body.innerHTML = "";
    let total = 0;

    cart.forEach((item) => {
      const row = document.createElement("tr");
      const itemTotal = item.price * item.qty;
      total += itemTotal;

      row.innerHTML = `
        <td>${item.name}</td>
        <td>${item.qty}</td>
        <td>${item.price.toLocaleString("fa-IR")} تومان</td>
        <td>${itemTotal.toLocaleString("fa-IR")} تومان</td>
        <td>
          <button class="btn btn-ghost remove-from-cart" data-id="${item.id}">حذف</button>
        </td>
      `;

      body.appendChild(row);
    });

    if (summaryTotalEl) {
      summaryTotalEl.textContent = total.toLocaleString("fa-IR") + " تومان";
    }
  }

  function renderComparePage() {
    const grid = document.getElementById("compare-grid");
    const emptyEl = document.getElementById("compare-empty");
    if (!grid || !emptyEl) return;

    const list = getCompare();

    if (!list.length) {
      emptyEl.classList.remove("hidden");
      grid.classList.add("hidden");
      grid.innerHTML = "";
      return;
    }

    emptyEl.classList.add("hidden");
    grid.classList.remove("hidden");
    grid.innerHTML = "";

    list.forEach((item) => {
      const card = document.createElement("div");
      card.className = "compare-card";
      card.innerHTML = `
        <h3>${item.name}</h3>
        <p class="product-price">${item.price.toLocaleString("fa-IR")} تومان</p>
        <button class="btn btn-ghost remove-from-compare" data-id="${item.id}">حذف از مقایسه</button>
      `;
      grid.appendChild(card);
    });
  }

  function initDashboardTabs() {
    const tabs = document.querySelectorAll(".dash-tab");
    const panels = {
      overview: document.getElementById("panel-overview"),
      profile: document.getElementById("panel-profile"),
      password: document.getElementById("panel-password"),
    };

    if (!tabs.length) return;

    tabs.forEach((tab) => {
      tab.addEventListener("click", () => {
        tabs.forEach((t) => t.classList.remove("active"));
        tab.classList.add("active");
        const target = tab.dataset.target;
        Object.keys(panels).forEach((key) => {
          if (panels[key]) {
            panels[key].classList.toggle("hidden", key !== target);
          }
        });
      });
    });
  }

  function highlightActiveNav() {
    const navLinks = document.querySelectorAll(".main-nav a");
    const path = window.location.pathname.split("/").pop() || "index.html";
    navLinks.forEach((link) => {
      if (link.getAttribute("href") === path) {
        link.classList.add("active-link");
      }
    });
  }

  function initNavToggle() {
    const toggle = document.querySelector(".nav-toggle");
    const header = document.querySelector(".site-header");
    if (!toggle || !header) return;

    toggle.addEventListener("click", () => {
      header.classList.toggle("nav-open");
    });

    document.querySelectorAll(".main-nav a").forEach((link) => {
      link.addEventListener("click", () => {
        header.classList.remove("nav-open");
      });
    });
  }

  updateCartCount();
  renderCartPage();
  renderComparePage();
  initDashboardTabs();
  highlightActiveNav();
  initNavToggle();
});
